import numpy as np
from math import pi,sqrt


def scatterLimits(score,Limit:float)->tuple: 
    #plain min and max
    Mini =np.min(score)
    Maxi =np.max(score)
    
    #squise data within a predefined fraction of standard deviation
    if Limit !=None:
        Mean  =np.mean(score)
        StDev =np.std(score) 
        Mini =max(Mean-StDev*Limit,Mini)
        Maxi =min(Mean+StDev*Limit,Maxi)
        
    return Mini,Maxi


def rotVectors_0_90(Orients:int) ->list:
    vec =np.zeros((2,Orients))
    X =np.arange(Orients)
    vec[0,:] =np.cos(X*pi/(Orients-1)/2)
    vec[1,:] =np.sin(X*pi/(Orients-1)/2)
    
    return vec


# calculates anisotropy of two consequent PCA components: First and Second 
# by plotting their scatter plot, rotating itand comparing resulted histograms
def anisotropy(scores,First:int,Second=None, Whitening =False,AniParameters=None):    
    if AniParameters==None:
        Cell =20 # defines histogram binning
        Limit =3 # cut of distribution expressed in number of std. dev.
        Rots =18   # nmber of rotations
    else:
        Cell =AniParameters[0]
        Limit =AniParameters[1]
        Rots =AniParameters[20]
        
    if Second ==None: #two consequent scores
        Second =First+1
    Length =scores.shape[0] #number of pixels    
    score1 = scores[:,First:First+1] 
    score2 = scores[:,Second:Second+1]   
    score1.shape =(1,Length)
    score2.shape =(1,Length)
        
    #limits
    Mini1,Maxi1 =scatterLimits(score1,Limit)
    #print('Limit',Limit,'min',Mini1,'max',Maxi1)
    Scaling =1
    if Whitening ==True: #discard the difference in variance
        Mini2,Maxi2 =scatterLimits(score2,Limit)
        Scaling =(Maxi1/Maxi2 + Mini1/Mini2)/2 #scale approximately same deviations from zero
    
    Bins =int(Length/Cell) #number of bins in histogram 
    #such as a given number of points (Cell) fall into one pixel (at plain distribution)
    
    vecRotated =rotVectors_0_90(Rots+1) #unit vectors oriented from 0° to 90°

    coupleScores =np.zeros((Length,2))
    coupleScores[:,0] =score1 
    coupleScores[:,1] =score2*Scaling
    projections =np.dot(coupleScores,vecRotated) #projections to series of unit vectors
    
    hist2D =np.zeros((Rots+1,Bins)) #histograms for all projections
    for i in range(Rots+1):
        hist2D[i,:] =np.histogram(projections[:,i], range=(Mini1,Maxi1), bins=Bins)[0]

    histMean =hist2D.mean(0) #mean histogram
    hist2D -=histMean #deviations from mean
    hist2D =np.square(hist2D)   #squared deviations
    
    with np.errstate(divide='ignore', invalid='ignore'):
        hist1D = np.true_divide(hist2D,histMean)   #normalize on counts
        hist1D[hist1D == np.inf] = 0
        hist1D = np.nan_to_num(hist1D)
        
    Ani =hist1D.sum()   #sum of squared deviations
    Ani /=Bins          #normalize on Bins
    Ani /=(Rots+1)      #normalize on rotations number
    Ani -=1             #criterion -> ZERO   

    # for ideally isotropic scatter plot, this must be Zero
    return Ani


# input: PCA scores with shape (m,p), m - number of pixels, p -number of extracted PCA components
def anisotropy_plot(scores,Whitening =False,AniParameters=None):
    Pairs =scores.shape[1]-1 #number of couple is one less than number of scores
    
    plot =np.zeros(Pairs)   
    for i in range(Pairs):
        Anisotropy =anisotropy(scores,i,Whitening =Whitening,AniParameters=AniParameters)
        #print(i+1,Anisotropy)
        plot[i] =Anisotropy
        
    return plot


# likelihood (non-normalized) that all non-zero anisotropy above the cut is noise
def cut_likelihood(anisotropyPlot,Cut,min_noise):
    Lambda=1.0e-16 #regularization parameter
    NumbComp =anisotropyPlot.shape[0]
    if Cut >(NumbComp-min_noise):
        print('Cut too high !!')
        return 0
    NoiseMid =np.mean(anisotropyPlot[Cut:])
    NoiseVar =np.var(anisotropyPlot[Cut:])
    Sigma = sqrt(NoiseVar)
    like = np.exp(-((anisotropyPlot-NoiseMid)/2/Sigma)**2)
    like[:Cut] =1-like[:Cut]
    like =np.where(like <Lambda,Lambda,like)
    Likelihood =np.prod(like)
    
    return Likelihood,NoiseMid,Sigma


# simplest Bayesian inference for best anisotropy cut
def auto_cut(anisotropyPlot,min_noise=2):
    NumbComp =anisotropyPlot.shape[0]
    prob =np.zeros(NumbComp)
    noiseMid =np.zeros(NumbComp)
    noiseSigma =np.zeros(NumbComp)
    for i in range(NumbComp-min_noise):
        prob[i],noiseMid[i],noiseSigma[i] = cut_likelihood(anisotropyPlot,i,min_noise)
        #print(i,Prob[i])
    prob /=np.sum(prob)
    Best_Cut =np.argmax(prob)
    if Best_Cut ==0: Best_Cut =1
    NoiseMid =noiseMid[Best_Cut]
    NoiseSigma =noiseSigma[Best_Cut]
    
    return  Best_Cut,NoiseMid,NoiseSigma,prob     

